//
//  HeroUPlatform.h
//  HeroUSDK
//
//  Created by 魏太山 on 2020/10/26.
//  Copyright © 2020年 Hero. All rights reserved.

#import <HeroUSDK/HeroUProject.h>
#import <HeroUSDK/HeroUPay.h>
#import <HeroUSDK/HeroUData.h>
#import <HeroUSDK/HeroUBloc.h>
#import <HeroUSDK/HeroUShare.h>
#import <HeroUSDK/GameProject.h>
#import <UIKit/UIKit.h>

NS_ASSUME_NONNULL_BEGIN

//usdk信息类型
typedef NS_ENUM(NSInteger, HeroUInfoType){
    /**用户id*/
    HeroUInfoTypeUserId,
    /**用户名*/
    HeroUInfoTypeUserName,
    /**SDKId*/
    HeroUInfoTypeSDKId,
    /**渠道Id*/
    HeroUInfoTypeChannelId,
    /**渠道名称*/
    HeroUInfoTypeChannelName,
    /**设备号*/
    HeroUInfoTypeDeviceNum,

};

// 广告类型
typedef NS_ENUM(NSInteger, HeroUAdType){
    /**未知广告*/
    HeroUAdTypeUnKnow = -1,
    /**admob广告*/
    HeroUAdTypeAdmob = 1,
};

typedef NS_ENUM(NSInteger, HeroULanguageType) {
    /** 系统默认 */
    HeroULanguageTypeSystem = 0,
    /** 简体中文 */
    HeroULanguageTypeZhHans = 1,
    /** 繁体中文 */
    HeroULanguageTypeZhHant = 2,
    /** 英语 */
    HeroULanguageTypeEn = 3,
    /** 日语 */
    HeroULanguageTypeJa = 4,
    /** 韩语 */
    HeroULanguageTypeKo = 5,
};

@interface HeroUPlatform : NSObject

#pragma mark - 初始化方法
/*
 *  获取单例实例对象
 */
+ (HeroUPlatform *)sharedInstance;

/**
 HeroU初始化方法
 先初始化HeroUProject对象
 e.g.
 HeroUProject *project = [[HeroUProject alloc] init];
 project.productId = @"融合SDK服务器后台申请的产品id";
 project.productKey = @"融合SDK服务器后台申请的产品key";
 
 NSDictionary *params = @{@"gameId"  : @"",
                          @"productCode" : @"",
                          @"productKey" : @"",
                          @"projectId" : @"",
                          @"urlServer" : @"",
                          @"twitterKey" : @"",
                          @"twitterSecret" : @""
 
 }
 [project channelInitWithParameters:params];
 [[HeroUPlatform sharedInstance] initWithProject:project];
 @param project 项目参数对象
 */
- (void)initWithProject:(HeroUProject *)project;


/**  设置URL */
@property (nonatomic, copy) NSString *serverUrl DEPRECATED_MSG_ATTRIBUTE("v3.5.3以上建议调用initWithProject方法");

/**
 * 老的初始化方法，建议改为"initWithProject:"初始化方法
 @brief 设置项目参数
 @param project 项目设置参数对象
 @note 调用SDK之前必须保证GameProject对象的各项参数正确设置
 */
- (void)launchingPlatformWithApplication:(nullable UIApplication *)application
                                 project:(GameProject *)project
                             withOptions:(nullable NSDictionary *)launchOptions DEPRECATED_MSG_ATTRIBUTE("v3.5.3以上建议调用initWithProject方法");

#pragma mark - =================== 登录相关API ===================

/**
 @brief  调用SDK默认的登陆界面
 @note   如果开发者想自定义用户登陆界面，请调用后序用户相关的接口方法
 */
- (void)enterLoginView;

/**
 @brief  注销、退出登陆
 */
- (void)logout;

/**
 @brief  注销、退出登陆
 */
- (void)logoutAndSowLoginView;

#pragma mark - =================== 支付相关API ===================
/**
 @brief 调用SDKIAP内购
 @param paymentParametersData iap支付参数配置
 @result 0表示方法调用成功
 */
- (int)iapPurchaseWithData:(GamePaymentParameters *)paymentParametersData;

/**兑换礼品卡或代码*/
- (void)redeemGiftCardOrCodeWithData:(GamePaymentParameters *)data
                             success:(void (^)(NSDictionary *dic))success
                             failure:(void (^)(NSString *error))failure;

/*
 *  主动查询小票，进行补单，有界面。全球渠道生效
 */
- (void)queryTicket;
#pragma mark - =================== 数据上报API ===================

/** 设置基础数据 */
- (void)setBaseRoleInfoWithData:(HeroHDCBaseGameRoleInfo *)data;

/**  角色登录  */
- (void)roleLoginWithGameRoleInfo:(HeroHDCGameRoleInfo *)gameRoleInfo;

/**  角色注册  */
- (void)roleRegisterWithGameRoleInfo:(HeroHDCGameRoleInfo *)gameRoleInfo;

/**  角色升级  */
- (void)roleLevelUpWithGameRoleInfo:(HeroHDCGameRoleInfo *)gameRoleInfo;

/** 上报闪屏结束  */
- (void)postSplashScreenEndSuccess:(void (^)(id obj))success
                             faild:(void (^)(id obj))faild;

#pragma mark - =================== 其他API ===================

/**
 @brief 获取IDFA
 1.获取IDFA需要添加AppTrackingTransparency.framework系统库，并且设置为Optional
 2.在Info.plist添加NSUserTrackingUsageDescription权限描述
 不添加，可能导致iOS 14以上无法获取IDFA
 * @param completion 成功回调，成功后会返回
 1.firstIdfa: 首次安装的idfa(存入keychain)
 2.currentIdfa: 当前获取的idfa，iOS14以上用户拒绝权限会返回空字符串
 3.error: 错误信息，有error时，idfa会返回空串
 */
- (void)getIDFAWithCompletion:(void (^)(NSString* firstIdfa, NSString* currentIdfa, NSString* error))completion;
/**
 @brief 获取IDFV
 * @param completion 成功回调，成功后会返回
 1.firstIdfv: 首次安装的idfv(存入keychain)
 2.currentIdfv: 当前获取的idfv
 3.error: 错误信息，有error时，idfv会返回空串
 */
- (void)getIDFVWithCompletion:(void (^)(NSString* firstIdfv, NSString* currentIdfv, NSString* error))completion;

/**根据类型获取用户信息*/
- (NSString *)getInfoWithType:(HeroUInfoType)type;

/*
 *  打开用户中心
 */
- (void)showUserCenter;

/*
 *  清空用户存储
 */
- (void)cleanUserEntities;

/**
 @brief 获取登录用户名
 @result 若用户没登录，则返回nil
 */
- (NSString *)getUserName;

/**
 @brief 获取登录用户ID
 @result 若用户没登录，则返回nil
 */
- (NSString *)getUserId;

/**
 @brief 获取ID（部分游戏使用@"id"字段作为唯一标示符）
 @result 若用户没登录，则返回nil
 */
- (NSString *)getSdkId;

/**
 @brief 获取渠道ID，必须在初始化后调用
 */
- (NSString *)getChannelId;

/**
 @brief 获取渠道名称，必须在初始化后调用
 */
- (NSString *)getChannelName;

/**
 @brief 获取渠道SDK的版本名，必须在初始化后调用
 */
- (NSString *)getChannelSdkVersionName;

/**
 @brief 获取渠道的ProjectId，必须在初始化后调用
 */
- (NSString *)getProjectId;

/**
 *  设置Cmge SDK内部是否为公共版
 *  如果设置为YES, 那么为公共版
 *  如果设置为NO, 那么为非公共版
 */
- (void)setPublicLogin:(BOOL)publicLogin;


/**
 设置Cmge SDK是否为单机版
 
 @param isOnlineGame 如果设置YES,那么为单机版，如果设置为NO，那么为网游版
 
 */
- (void)setIsOnlineGame:(BOOL)isOnlineGame;

/**
 *  当前是否已经登录
 *
 *  @return YES 已经登录
 *          NO 没有登录
 */
- (BOOL)isLogined;

/**
 *  是否是纯净版，
 *
 *  @return YES 是纯净版本则会去除usdk相关业务，只保留渠道业务，服务器token校验需要使用新接口校验
 *          NO  不是纯净版本，不做出改变
 */
- (BOOL)isPureVersion;

/*
 * 获取设备号(优先取的IDFA、没取到则取的UUID)
 */
- (NSString *)getDeviceNum;


/** 扫码登录（在PC端登录,SDK需要登录状态）*/
- (void)showScanViewWithExt:(NSString *)exit;

/*
 * 打开客服界面
 */
- (void)openServiceView;

/**
下线回传
@param result 结果
*/
- (void)cpKickOffCallBackWithResult:(NSDictionary *)result ;

/*
 * 获取当前登录用户的信息接口
 */
- (void)requestUserInfoSuccess:(void (^)(id obj))success
                             faild:(void (^)(id obj))faild ;


/**获取首次弹窗协议内容
{
       childAgrName = "个人信息保护政策";
       childAgrUrl = "";
       priAgrName = "个人信息保护政策";
       priAgrUrl = "";
       userAgrName = "用户协议";
       userAgrUrl = "";
       version = 1;
};
 */
- (NSDictionary *)getProtocolResult;

/**
 cp自绘制协议弹窗界面，点击同意设置接口
 */
- (void)setAgreeProtocol;

/**
 * @abstract
 * 调用 track 接口，追踪一个带有属性的 event
 *
 * @discussion
 * propertyDict 是一个 Map。
 * 其中的 key 是 Property 的名称，必须是 NSString
 * value 则是 Property 的内容，只支持 NSString、NSNumber、NSSet、NSArray、NSDate 这些类型
 * 特别的，NSSet 或者 NSArray 类型的 value 中目前只支持其中的元素是 NSString
 *
 * @param event             event的名称
 * @param propertyDict     event的属性
 */
- (void)track:(NSString *)event withProperties:(nullable NSDictionary *)propertyDict;

/**
 获取全埋点数据
 */
- (void)trackEventDataCallback:(void (^)(NSDictionary *data))callback;


/**将图片保存至相册
 * @param imageObj 支持 UIImage图片对象，NSData图片数据 和 NSString图片绝对路径 3种类型，传任意一种即可
 * @param albumName 相册名称，传@""或者nil则保存到默认相册
 */
- (void)addPicToAlbumWithImageObj:(id)imageObj
                        albumName:(NSString *)albumName
                          success:(void (^)(NSDictionary *result))success
                          failure:(void (^)(NSString *error))failure;
    

#pragma mark - =================== 全球渠道相关API ===================
/*
 *  账号删除
 *  调用此接口会弹出删除账号界面，删除账号后服务器也会删除，请谨慎操作
 */
- (void)accountCancellation;

/**
 *  解除【账号删除】状态，将账号恢复成正常账号
 */
- (void)accountUnCancellationWithSuccess:(void (^)(id obj))success
                                 failure:(void (^)(id obj))failure;

/**
 * 上报问题
 */
- (void)uploadIssue;

/*
 *  打开Xsolla商城支付
 *  @param HeroUXsollaModel  参数模型
 */
- (void)startXsollaPayWithXsollaModel:(HeroUXsollaModel *)xsollaModel;

/*
 *  全球渠道绑定接口
 *  1  =   Facebook
 *  2  =   Google
 *  3  =   GameCenter
 *  4  =   WeChat
 *  5  =   Twitter
 *  7  =   Apple
 *  8  =   Line
 */
- (void)bindAccountWithType:(NSInteger)bindType
                  success:(void (^)(NSDictionary *result))success
                    failure:(void (^)(NSString *error))failure;

/*
 *  获取全球渠道用户绑定状态
 *  result数据  {
 "huaweiBind" : 0,
 "weChatBind" : 0,
 "gameWithBind" : 0,
 "twitterBind" : 0,
 "appleBind" : 0,
 "code" : 0,
 "fbBind" : 0,
 "lineBind" : 0,
 "oppoBind" : 0,
 "pwdStatus" : 0,
 "msg" : "success",
 "googleBind" : 0,
 "gcBind" : 0
}
 */
- (void)getBindStateWithSuccess:(void (^)(NSDictionary *result))success
                        failure:(void (^)(NSString *error))failure;

/** 绑定邮箱功能 */
- (void)bindEmailWithSuccess:(void (^)(NSDictionary *result))success
                     failure:(void (^)(NSString *error))failure;

/**
 设置SDK语言
 **/
- (void)setLanguageType:(HeroULanguageType)languageType;

/** 获取苹果商品列表**/
- (void)getAppleSkuDetailsList:(NSArray<NSString *> *)gameGoodsList
                       success:(void (^)(NSArray<NSString *> *))success
                       failure:(void (^)(NSString *error))failure;
    
#pragma mark - =================== 助手相关API ===================
/*
 *  java 服务器，请求配置参数等
 */
@property (nonatomic, copy) NSString *javaServer DEPRECATED_MSG_ATTRIBUTE("v4.5.1方法不可用");
/*
 *  php服务器，打开网页，红点等
 */
@property (nonatomic, copy) NSString *phpServer DEPRECATED_MSG_ATTRIBUTE("v4.5.1方法不可用");

/**
 初始化助手模块
 在此方法中调用- (BOOL)application:(UIApplication *)application didFinishLaunchingWithOptions:(NSDictionary *)launchOptions
 */
-(void)setBlocConfig:(HeroUBlocConfig *)config DEPRECATED_MSG_ATTRIBUTE("v4.5.1方法不可用");

/*
 *  打开指定页面回调
 */
@property (nonatomic, copy  ) void (^onSdkRedirectMessage)(NSString *action,id data) DEPRECATED_MSG_ATTRIBUTE("v4.5.1方法不可用");
/*
 *  设置助手浮标位置
 */
@property (nonatomic, assign) CGPoint floatBtnFrame DEPRECATED_MSG_ATTRIBUTE("v4.5.1方法不可用");

/*
 *  是否显示小恐龙浮标
 */
- (void)needShowPhoneFloat:(BOOL)needShow DEPRECATED_MSG_ATTRIBUTE("v4.5.1方法不可用");

/*
 *  =================== 攻略、赛事、社区模块（根据业务需要接入） ===================
 */

/**
 *  上报比赛信息
 *  BlocMatchData
 *  @param matchData 比赛信息数据模型
 */
- (void)postMatch:(NSDictionary *)matchData DEPRECATED_MSG_ATTRIBUTE("v4.5.1方法不可用");

/**
 是否显示入口
 
 @param key 入口对应key值 （攻略：tactic  赛事：match  拉新：spread 商城：shop）对应值可咨询用户中心
 
 @return YES(是，显示) NO(否，隐藏)
 */
- (BOOL)showEntrance:(NSString *)key ;

/**
 *  打开专区
 *
 *  @param action    action(必填)
 *  @param data      data(可选)
 *  @param module    必填，可根据对照表填写
 *  @param exitBlock 退出回调
 */
- (void)openForumViewWithAction:(NSString *)action
                           data:(NSString *)data
                         module:(NSString *)module
                           exit:(void (^)(void))exitBlock ;
/*
 *  直接传URL打开专区
 *
 *  @param openForumViewWithUrl 是否需要拼接公共参数
 *  @param exitBlock 退出回调
 */
- (void)openForumViewWithUrl:(NSString *)urlStr
        needCommonParamiters:(BOOL)needCommonParamiters
                        exit:(void (^)(void))exitBlock ;

/**
 *  主动获取红点信息（被动使用通知 BLOC_TAG_NOTIFICATION_RED）
 *
 *  @param resBlock 获取红点的回调
 */
- (void)checkRedPoint:(void(^)(void))resBlock DEPRECATED_MSG_ATTRIBUTE("v4.5.1方法不可用");

/**
 根据key值是否显示红点
 在通知中调用或- (void)checkRedPoint:(void(^)())resBlock;回调中使用
 @param key key
 
 @return yes为显示 no为隐藏
 */
- (BOOL)showRed:(NSString *)key DEPRECATED_MSG_ATTRIBUTE("v4.5.1方法不可用");

/*
 *  =================== 关系链模块 ===================
 */

/**
 打开邀请密友列表页面
 */
- (void)openRelationChainView DEPRECATED_MSG_ATTRIBUTE("v4.5.1方法不可用");


/**
 打开邀请密友页面
 */
- (void)openAddressBookView DEPRECATED_MSG_ATTRIBUTE("v4.5.1方法不可用");


/**
 密友列表：监测密友状态是否已经激活
 chainState = YES，已经激活
 chainState = NO，没有激活
 
 @param chainState chainState，BOOL值
 */
- (void)checkChainState:(void(^)(BOOL state))chainState DEPRECATED_MSG_ATTRIBUTE("v4.5.1方法不可用");

#pragma mark - =================== 分享相关API ===================

/*
 *  初始化QQ
 */
- (void)initWithQQkey:(NSString *)key universalLink:(NSString *)universalLink;
/*
 *  初始化微信
 */
- (void)initWithWechatKey:(NSString *)key universalLink:(NSString *)universalLink;
/**
 *  是否安装QQ
 */
- (BOOL)isQQInstalled;
/**
 *  是否安装微信
 */
- (BOOL)isWXAppInstalled;
/*
 * 有分享界⾯面 - shareModels为分享数据模型，包含分享链接、描述、分享类型等必要参数
 * success成功回调 failure失败回调
 */
- (void)showShareViewWithShareDataArray:(NSArray *)shareModels
                                success:(void(^)(NSString *successMessage,HeroUShareStatus status,HeroUShareTaget target))success
                                failure:(void(^)(NSString *failureMessage,HeroUShareStatus status,HeroUShareTaget target))failure;
/*
 * 分享--⽆分享界⾯面
 * shareModel为分享数据模型，包含分享链接、描述、分享类型等必要参数
 * success成功回调 failure失败回调 */
- (void)shareWithShareData:(HeroUShareModel *)shareModel
                   success:(void(^)(NSString *successMessage,HeroUShareStatus status,HeroUShareTaget target))success
                   failure:(void(^)(NSString *failureMessage,HeroUShareStatus status,HeroUShareTaget target))failure;

/**
 三方登录回调
 */
- (BOOL)heroU_application:(UIApplication *)application
      handleOpenURL:(NSURL *)url ;

/*
*  三方登录返回来的回调
*/
- (BOOL)handleOpenUniversalLink:(NSUserActivity *)userActivity ;

#pragma mark - =================== 新分享相关API ===================

/*
 * 分享
 * @param shareModel 分享数据模型，包含分享链接、描述、分享类型等必要参数
 * @param success 成功回调
 * @param failure 失败回调
 */
- (void)heroUShareWithShareModel:(HeroUShareModel *)shareModel
                         success:(void(^)(NSString *successMessage,HeroUShareStatus status,HeroUShareTaget target))success
                         failure:(void(^)(NSString *failureMessage,HeroUShareStatus status,HeroUShareTaget target))failure;


#pragma mark - =================== 浏览器 ===================

/*
 * 通过webId打开浏览器
 * @param webId   支撑平台配置的ID
 * @param success 成功回调
 * @param failure 失败回调
 */
- (void)heroUOpenBrowserByWebId:(NSString *)webId
                         success:(void (^)(NSDictionary *dic))success
                         failure:(void (^)(NSString *error))failure;

/*
 * 通过webId打开浏览器
 * @param webId       支撑平台配置的ID
 * @param success     成功回调
 * @param failure     失败回调
 * @param closeHandle 关闭回调
 */
- (void)heroUOpenBrowserByWebId:(NSString *)webId
                        success:(void (^)(NSDictionary *dic))success
                        failure:(void (^)(NSString *error))failure
                    closeHandle:(void (^)(void))closeHandle;

/*
 * 通过url打开浏览器
 * @param url          http或者https链接
 * @param orientation  0为横屏，1为竖屏
 * @param bts          按钮数组，传nil，默认左上角有返回按钮
 * @param success      成功回调
 * @param failure      失败回调
 */
- (void)heroUOpenBrowserByUrl:(NSString *)url
                  orientation:(NSInteger)orientation
                          bts:(NSArray<NSDictionary *> *)bts
                      success:(void (^)(NSDictionary *dic))success
                      failure:(void (^)(NSString *error))failure;

/**
 * 获取所有浏览器对应webId的开关
 * 返回值为NSDictionary，key为webId；value，1为开启，0为关闭，如果没有key，则返回空字典
 *  {
        "test02" : 1,
        "test03" : 1
    }
 */
- (NSDictionary *)getWebSwitchCase;

#pragma mark - =================== 市场SDK插件 ===================

/**
 @brief 系统回调接口，必接
*/
- (void)applicationDidBecomeActive:(UIApplication *)application ;

/**
 @brief 自定义事件追踪接口，融合接口
 @param eventName  事件名称
 @param parameters 扩展字典，默认可以传nil
*/
- (void)logEvent:(NSString *)eventName
      parameters:(NSDictionary *)parameters;

/**
 @brief 购买事件追踪，融合接口
 @param orderId    订单Id
 @param amount     金额
 @param currency   货币
 @param parameters 扩展字典，默认可以传nil
 */
- (void)logPurchaseWithOrderId:(NSString *)orderId
                        amount:(double)amount
                      currency:(NSString *)currency
                    parameters:(NSDictionary *)parameters;

/**
 @brief fireBase_邮箱归因
 */
- (void)sendFirebaseEmailAddress:(NSString *)emailAddress ;

/**
 @brief 初始化设置appsFlyerCustomerUserID可能过早，还没有对应的值，可以选择在需要的时候调用此接口设置 appsFlyerCustomerUserID
 */
- (void)setAppsFlyerCustomerUserID:(NSString *)customerUserID ;

/**
 @brief appsflyer 获取onelink点击链接内容
 @param callback    内容
 */
- (void)getAppsFlyerdidResolveDeepLinkDataCallback:(void (^)(NSDictionary *result))callback;

/**
 @brief appsflyer 获取 OnConversionDataSuccess 回调内容
 `conversionInfo` contains information about install.
 Organic/non-organic, etc.  conversionInfo May contain <code>null</code> values for some keys. Please handle this case.
 */
- (void)getAppsFlyerOnConversionDataSuccess:(void (^)(NSDictionary *conversionInfo))callback;

/**
 @brief appsflyer 获取 OnConversionDataFail 回调内容
 Any errors that occurred during the conversion request.
 */
- (void)getAppsFlyeOnConversionDataFail:(void (^)(NSError *error))callback;

/**
 @brief appsflyer 获取 OnAppOpenAttribution 回调内容
 `attributionData` contains information about OneLink, deeplink.
 */
- (void)getAppsFlyerOnAppOpenAttribution:(void (^)(NSDictionary *attributionData))callback;

/**
 @brief appsflyer 获取 OnAppOpenAttributionFailure 回调内容
 Any errors that occurred during the attribution request.
 */
- (void)getAppsFlyerOnAppOpenAttributionFailure:(void (^)(NSError *error))callback;

/**
 @brief 显示激励广告
 @param adId    激励视频广告位ID
 @param type    广告平台，暂时只支持admob
 @param successHandler    成功回调，视频播放完成，发送奖励会触发该回调
 @param failureHandler    失败回调，视频加载失败是会触发该回调
 @param closeHandler      关闭回调，关闭视频时会触发该回调
 */
- (void)showAdVideoWithAdId:(NSString *)adId
                       type:(HeroUAdType)type
             successHandler:(void (^)(NSString *type, double amount))successHandler
             failureHandler:(void (^)(NSString *msg, NSInteger code))failureHandler
               closeHandler:(void (^)(void))closeHandler;

/**
 @brief 显示激励广告-新接口
 @param adId    激励视频广告位ID
 @param type    广告平台，暂时只支持admob
 @param successHandler    成功回调，视频播放完成，发送奖励会触发该回调
 @param failureHandler    失败回调，视频加载失败是会触发该回调
 @param showHandler       显示回调，视频刚开始要显示时会触发该回调
 @param closeHandler      关闭回调，关闭视频时会触发该回调
 */
- (void)showAdVideoWithAdId:(NSString *)adId
                       type:(HeroUAdType)type
             successHandler:(void (^)(NSString *type, double amount))successHandler
             failureHandler:(void (^)(NSString *msg, NSInteger code))failureHandler
                showHandler:(void (^)(void))showHandler
               closeHandler:(void (^)(void))closeHandler;

/**
 @brief CMP许可回调
 @param callback 回调
 */
- (void)withOnCmpButtonClickedCallback:(void (^)(void))callback;

#pragma mark - =================== AIHelp插件 ===================

/**
 @brief  打开AIHelp对话框
 */
- (void)showAIHelpConversation;

#pragma mark - =================== 巨量SDK插件 ===================

/**
 @brief  启动巨量SDK插件，需要配置scheme，scheme规则为应用包名
 */
- (void)startBDASDK;

- (NSString *)getBDAClickId;

#pragma mark - =================== 抖音双通SDK插件 ===================

/**
 @brief  启动抖音双通SDK插件，需要配置scheme，scheme规则为dygame${app_id}
 */
- (void)startDouYinUnionSDK;

@end

NS_ASSUME_NONNULL_END
